import sys, binascii, logging

class db(object):
    def __init__(self, filename="values.db"):
        self.filename = filename
        self.logger = logging.getLogger("valueslib")
        self.bases = {}
        self.pairs = {}
        self.reversepairs = {}
        self.dummies = {}
        self.extras = {}

        data = open(self.filename, "rb").read()
        if len(data) != 0:
            for entry in data.split("\n"):
                if entry[0:4] == "BASE":
                    base = entry[5:]
                    if base in self.bases:
                        self.logger.error("Duplicate base %s", base)
                        sys.exit()
                    self.bases[base] = True
                elif entry[0:4] == "PAIR":
                    (_, cha, resp) = entry.split(" ")
                    if cha in self.pairs or cha in self.dummies or resp in self.reversepairs or resp in self.extras:
                        self.logger.error("Duplicate challenge/response %s %s", cha, resp)
                    self.pairs[cha] = resp
                    self.reversepairs[resp] = cha
                elif entry[0:4] == "DUMM":
                    dummy = entry[5:]
                    if dummy in self.dummies or dummy in self.pairs:
                        self.logger.error("Duplicate dummy %s", dummy)
                        sys.exit()
                    self.dummies[dummy] = True
                elif entry[0:4] == "EXTR":
                    extra = entry[5:]
                    if extra in self.extras or extra in self.reversepairs:
                        self.logger.error("Duplicate extra %s", extra)
                        sys.exit()
                    self.extras[extra] = True
                elif len(entry) == 0:
                    pass
                else:
                    self.logger.error("Invalid entry!")
                    sys.exit()

    def add_dummy(self, dummy):
        if dummy not in self.pairs and dummy not in self.dummies:
            self.logger.debug("Adding dummy %s", dummy)
            self.dummies[dummy] = True
            
    def add_extra(self, extra):
        if extra not in self.reversepairs and extra not in self.extras:
            self.logger.debug("Adding extra %s", extra)
            self.extras[extra] = True
            
    def add_base(self, base):
        if base not in self.bases:
            self.logger.debug("Adding base %s", base)
            self.bases[base] = True
    
    def add_cha_resp(self, cha, resp):
        if cha in self.pairs:
            if self.pairs[cha] == resp:
                # pair already there
                return
            else:
                self.logger.error("Duplicate response for a challenge! %s %s %s", cha, self.pairs[cha], resp)
                sys.exit()
        if resp in self.reversepairs:
            if self.reversepairs[resp] == cha:
                # pair already there
                return
            else:
                self.logger.error("Duplicate challenge for a response! %s %s %s", resp, self.reversepairs[resp], cha)
                sys.exit()
        if cha in self.dummies:
            del self.dummies[cha]
            self.logger.debug("Removed from dummies, good!")
        if resp in self.extras:
            del self.extras[resp]
            self.logger.debug("Removed from extras, good!")
        self.pairs[cha] = resp
        self.reversepairs[resp] = cha
        self.logger.debug("Added new pair %s %s", cha, resp)
    
    def write(self):
        output = []
        for base in sorted(self.bases.keys()):
            output.append("BASE " + base)
        for cha in sorted(self.pairs.keys()):
            output.append("PAIR " + cha + " " + self.pairs[cha])
        for dummy in sorted(self.dummies.keys()):
            output.append("DUMM " + dummy)
        for extra in sorted(self.extras.keys()):
            output.append("EXTR " + extra)
        open(self.filename, "wb").write("\n".join(output))

